<?php
$contentPath = __DIR__ . '/data/content.json';
$message = null;
$errors = [];

function h($value): string {
    return htmlspecialchars((string)($value ?? ''), ENT_QUOTES, 'UTF-8');
}

function format_json($value): string {
    return json_encode($value, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
}

function decode_json_field(string $name, string $label, array &$errors)
{
    $raw = $_POST[$name] ?? '';
    $decoded = json_decode($raw, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        $errors[] = $label . ' : ' . json_last_error_msg();
        return $raw; // conserve la saisie pour ré-affichage
    }
    return $decoded;
}

function value_from(array $source, array $path, $default = '')
{
    $carry = $source;
    foreach ($path as $key) {
        if (!is_array($carry) || !array_key_exists($key, $carry)) {
            return $default;
        }
        $carry = $carry[$key];
    }
    return $carry;
}

$contentData = @file_get_contents($contentPath);
if ($contentData === false) {
    http_response_code(500);
    exit('Impossible de charger le fichier de configuration.');
}

$content = json_decode($contentData, true);
if (!is_array($content)) {
    http_response_code(500);
    exit('Fichier de configuration JSON invalide.');
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Site
    $newContent = [];
    $newContent['site'] = [
        'title' => trim($_POST['site_title'] ?? ''),
        'meta_description' => trim($_POST['site_meta_description'] ?? ''),
        'theme_color' => trim($_POST['site_theme_color'] ?? ''),
        'og' => [
            'type' => trim($_POST['site_og_type'] ?? ''),
            'title' => trim($_POST['site_og_title'] ?? ''),
            'description' => trim($_POST['site_og_description'] ?? ''),
            'image' => trim($_POST['site_og_image'] ?? ''),
            'url' => trim($_POST['site_og_url'] ?? ''),
        ],
    ];

    // Navigation
    $navItems = extract_array_items('nav_item', ['label', 'href']);
    $newContent['navigation'] = [
        'brand_label' => trim($_POST['nav_brand_label'] ?? ''),
        'brand_aria' => trim($_POST['nav_brand_aria'] ?? ''),
        'items' => $navItems,
        'cta' => [
            'label' => trim($_POST['nav_cta_label'] ?? ''),
            'href' => trim($_POST['nav_cta_href'] ?? ''),
        ],
    ];

    // Hero
    $heroStats = extract_array_items('hero_stat', ['value', 'label']);
    $heroCtas = extract_array_items('hero_cta', ['style', 'label', 'href', 'icon']);
    $heroHighlights = [];
    $idx = 0;
    while (isset($_POST['hero_highlight_' . $idx])) {
        $text = trim($_POST['hero_highlight_' . $idx] ?? '');
        if (!empty($text)) {
            // Si c'est juste une string
            if (empty($_POST['hero_highlight_' . $idx . '_icon'])) {
                $heroHighlights[] = $text;
            } else {
                $heroHighlights[] = [
                    'icon' => trim($_POST['hero_highlight_' . $idx . '_icon'] ?? ''),
                    'text' => $text,
                ];
            }
        }
        $idx++;
    }
    $newContent['hero'] = [
        'chip' => trim($_POST['hero_chip'] ?? ''),
        'title' => trim($_POST['hero_title'] ?? ''),
        'subtitle' => trim($_POST['hero_subtitle'] ?? ''),
        'stats' => $heroStats,
        'ctas' => $heroCtas,
        'share_label' => trim($_POST['hero_share_label'] ?? ''),
        'share_message_prefix' => trim($_POST['hero_share_prefix'] ?? ''),
        'about' => [
            'subtitle' => trim($_POST['hero_about_subtitle'] ?? ''),
            'title' => trim($_POST['hero_about_title'] ?? ''),
            'description' => trim($_POST['hero_about_description'] ?? ''),
            'highlights' => $heroHighlights,
            'cta' => [
                'label' => trim($_POST['hero_about_cta_label'] ?? ''),
                'href' => trim($_POST['hero_about_cta_href'] ?? ''),
                'icon' => trim($_POST['hero_about_cta_icon'] ?? ''),
            ],
        ],
    ];

    // Portrait
    $portraitAchievements = extract_array_items('portrait_achievement', ['icon', 'text']);
    $portraitStats = extract_array_items('portrait_stat', ['value', 'label']);
    $portraitConvictions = extract_array_items('portrait_conviction', ['title', 'text']);
    $newContent['portrait'] = [
        'title' => trim($_POST['portrait_title'] ?? ''),
        'description' => trim($_POST['portrait_description'] ?? ''),
        'image_alt' => trim($_POST['portrait_image_alt'] ?? ''),
        'achievements' => $portraitAchievements,
        'stats' => $portraitStats,
        'convictions' => [
            'heading' => trim($_POST['portrait_convictions_heading'] ?? ''),
            'items' => $portraitConvictions,
        ],
    ];

    // Project
    $projectSegments = extract_array_items('project_segment', ['title', 'text']);
    $projectCtas = extract_array_items('project_cta', ['style', 'label', 'href', 'icon']);
    $projectCards = extract_array_items('project_card', ['icon', 'title', 'text']);
    $newContent['project'] = [
        'eyebrow' => trim($_POST['project_eyebrow'] ?? ''),
        'title' => trim($_POST['project_title'] ?? ''),
        'description' => trim($_POST['project_description'] ?? ''),
        'segments' => $projectSegments,
        'ctas' => $projectCtas,
        'cards' => $projectCards,
    ];

    // Vision
    $visionCards = extract_array_items('vision_card', ['title', 'text']);
    $visionSpecific = extract_array_items('vision_specific', ['title', 'text']);
    $newContent['vision'] = [
        'eyebrow' => trim($_POST['vision_eyebrow'] ?? ''),
        'title' => trim($_POST['vision_title'] ?? ''),
        'subtitle' => trim($_POST['vision_subtitle'] ?? ''),
        'objective' => [
            'title' => trim($_POST['vision_objective_title'] ?? ''),
            'text' => trim($_POST['vision_objective_text'] ?? ''),
        ],
        'cards' => $visionCards,
        'specific_title' => trim($_POST['vision_specific_title'] ?? ''),
        'specific_items' => $visionSpecific,
    ];

    // Piliers
    $pillarsItems = extract_array_items('pillar', ['icon', 'title', 'text']);
    $newContent['pillars'] = [
        'eyebrow' => trim($_POST['pillars_eyebrow'] ?? ''),
        'title' => trim($_POST['pillars_title'] ?? ''),
        'subtitle' => trim($_POST['pillars_subtitle'] ?? ''),
        'items' => $pillarsItems,
    ];

    // Activités - Les blocs ont une structure plus complexe avec items
    $activitiesBlocks = [];
    $idx = 0;
    while (isset($_POST['activity_block_' . $idx . '_title'])) {
        $block = [
            'title' => trim($_POST['activity_block_' . $idx . '_title'] ?? ''),
            'col' => trim($_POST['activity_block_' . $idx . '_col'] ?? 'col-md-12'),
        ];
        // Extraire les items du bloc
        $items = [];
        $itemIdx = 0;
        while (isset($_POST['activity_block_' . $idx . '_item_' . $itemIdx])) {
            $item = trim($_POST['activity_block_' . $idx . '_item_' . $itemIdx] ?? '');
            if (!empty($item)) {
                $items[] = $item;
            }
            $itemIdx++;
        }
        $block['items'] = $items;
        if (!empty($block['title']) || !empty($items)) {
            $activitiesBlocks[] = $block;
        }
        $idx++;
    }

    // Manifeste
    $manifesteCtas = extract_array_items('manifeste_cta', ['style', 'label', 'href', 'icon']);
    $newContent['manifeste'] = [
        'eyebrow' => trim($_POST['manifeste_eyebrow'] ?? ''),
        'title' => trim($_POST['manifeste_title'] ?? ''),
        'description' => trim($_POST['manifeste_description'] ?? ''),
        'ctas' => $manifesteCtas,
        'image_alt' => trim($_POST['manifeste_image_alt'] ?? ''),
    ];

    // Agenda
    $agendaTimeline = extract_array_items('agenda_timeline', ['title', 'text']);
    $agendaFields = extract_array_items('agenda_field', ['id', 'name', 'label', 'type', 'placeholder', 'required']);
    $newContent['agenda'] = [
        'eyebrow' => trim($_POST['agenda_eyebrow'] ?? ''),
        'title' => trim($_POST['agenda_title'] ?? ''),
        'timeline' => $agendaTimeline,
        'form' => [
            'heading' => trim($_POST['agenda_form_heading'] ?? ''),
            'fields' => $agendaFields,
            'submit_label' => trim($_POST['agenda_form_submit'] ?? ''),
            'disclaimer' => trim($_POST['agenda_form_disclaimer'] ?? ''),
        ],
    ];

    // Médias
    $mediaItems = extract_array_items('media_item', ['image', 'alt', 'title', 'text', 'href', 'link_label']);
    $newContent['media'] = [
        'eyebrow' => trim($_POST['media_eyebrow'] ?? ''),
        'title' => trim($_POST['media_title'] ?? ''),
        'subtitle' => trim($_POST['media_subtitle'] ?? ''),
        'items' => $mediaItems,
    ];

    // Témoignages
    $testimonialsItems = extract_array_items('testimonial', ['quote', 'author']);
    $newContent['testimonials'] = [
        'eyebrow' => trim($_POST['testimonials_eyebrow'] ?? ''),
        'title' => trim($_POST['testimonials_title'] ?? ''),
        'items' => $testimonialsItems,
    ];

    // FAQ
    $faqItems = extract_array_items('faq', ['question', 'answer']);
    $newContent['faq'] = [
        'eyebrow' => trim($_POST['faq_eyebrow'] ?? ''),
        'title' => trim($_POST['faq_title'] ?? ''),
        'items' => $faqItems,
    ];

    // Contact
    $contactFormFields = extract_array_items('contact_form_field', ['id', 'name', 'label', 'type', 'placeholder', 'required']);
    $contactCardItems = extract_array_items('contact_card_item', ['icon', 'text', 'href']);
    $contactSocial = extract_array_items('contact_social', ['icon', 'href', 'aria_label']);
    $newContent['contact'] = [
        'eyebrow' => trim($_POST['contact_eyebrow'] ?? ''),
        'title' => trim($_POST['contact_title'] ?? ''),
        'subtitle' => trim($_POST['contact_subtitle'] ?? ''),
        'form' => [
            'fields' => $contactFormFields,
            'submit_label' => trim($_POST['contact_form_submit'] ?? ''),
            'disclaimer' => trim($_POST['contact_form_disclaimer'] ?? ''),
        ],
        'contacts_card' => [
            'title' => trim($_POST['contact_card_title'] ?? ''),
            'items' => $contactCardItems,
            'social_title' => trim($_POST['contact_social_title'] ?? ''),
            'social_links' => $contactSocial,
        ],
    ];

    // Footer et divers
    $newContent['footer'] = [
        'copyright' => trim($_POST['footer_copyright'] ?? ''),
        'slogan' => trim($_POST['footer_slogan'] ?? ''),
    ];
    $newContent['forms'] = [
        'mail_subject' => trim($_POST['forms_mail_subject'] ?? ''),
    ];

    if (empty($errors)) {
        $encoded = format_json($newContent);
        if (file_put_contents($contentPath, $encoded, LOCK_EX) === false) {
            $errors[] = "Impossible d'écrire le fichier de configuration.";
        } else {
            $content = $newContent;
            $contentData = $encoded;
            $message = 'Contenu mis à jour avec succès.';
            $_POST = []; // nettoie pour réafficher les valeurs sauvegardées
        }
    }
}

$usePostValues = ($_SERVER['REQUEST_METHOD'] === 'POST' && !empty($errors));

function field(string $name, $default, bool $usePost)
{
    return $usePost ? ($_POST[$name] ?? $default) : $default;
}

// Helper pour extraire les éléments d'un tableau depuis POST
function extract_array_items(string $prefix, array $expectedKeys): array {
    $items = [];
    $index = 0;
    while (isset($_POST[$prefix . '_' . $index . '_' . $expectedKeys[0]])) {
        $item = [];
        $hasData = false;
        foreach ($expectedKeys as $key) {
            $fieldName = $prefix . '_' . $index . '_' . $key;
            $value = trim($_POST[$fieldName] ?? '');
            // Gérer les valeurs booléennes pour 'required'
            if ($key === 'required') {
                $item[$key] = !empty($value) && $value !== '0' && $value !== 'false';
            } else {
                $item[$key] = $value;
            }
            if (!empty($value)) {
                $hasData = true;
            }
        }
        if ($hasData) {
            $items[] = $item;
        }
        $index++;
    }
    return $items;
}

// Helper pour générer un formulaire d'éléments répétitifs
function render_array_items(string $id, array $items, array $fields, string $itemLabel): string {
    $html = '<div class="array-items-container" id="' . h($id) . '_container">';
    $index = 0;
    foreach ($items as $item) {
        $html .= render_array_item($id, $index, $fields, $item, $itemLabel);
        $index++;
    }
    $html .= '</div>';
    $html .= '<button type="button" class="btn btn-sm btn-outline-primary mt-2" onclick="addArrayItem(\'' . h($id) . '\', ' . h(json_encode($fields)) . ', \'' . h($itemLabel) . '\')">';
    $html .= '<i class="bi bi-plus-circle"></i> Ajouter un élément</button>';
    return $html;
}

function render_array_item(string $id, int $index, array $fields, $item = [], string $itemLabel = ''): string {
    // Normaliser l'item : si c'est une string, la convertir en array avec la première clé des fields
    if (is_string($item)) {
        $firstField = $fields[0] ?? null;
        if ($firstField) {
            $item = [$firstField['name'] => $item];
        } else {
            $item = [];
        }
    }
    if (!is_array($item)) {
        $item = [];
    }
    
    $html = '<div class="array-item-card card mb-2" data-index="' . $index . '">';
    $html .= '<div class="card-body">';
    $html .= '<div class="d-flex justify-content-between align-items-center mb-2">';
    $html .= '<small class="text-muted fw-bold">' . ($itemLabel ?: 'Élément #' . ($index + 1)) . '</small>';
    $html .= '<button type="button" class="btn btn-sm btn-outline-danger" onclick="removeArrayItem(this)"><i class="bi bi-trash"></i></button>';
    $html .= '</div>';
    $html .= '<div class="row g-2">';
    foreach ($fields as $field) {
        $fieldName = $id . '_' . $index . '_' . $field['name'];
        $fieldValue = $item[$field['name']] ?? $field['default'] ?? '';
        $html .= '<div class="' . ($field['col'] ?? 'col-12') . '">';
        $html .= '<label class="form-label small">' . h($field['label']) . '</label>';
        if ($field['type'] === 'textarea') {
            $html .= '<textarea name="' . h($fieldName) . '" class="form-control form-control-sm" rows="' . ($field['rows'] ?? 2) . '" placeholder="' . h($field['placeholder'] ?? '') . '">' . h($fieldValue) . '</textarea>';
        } elseif ($field['type'] === 'select') {
            $html .= '<select name="' . h($fieldName) . '" class="form-select form-select-sm">';
            foreach ($field['options'] ?? [] as $optValue => $optLabel) {
                $selected = ($fieldValue === $optValue || $fieldValue === $optLabel || (empty($fieldValue) && $optValue === ($field['default'] ?? ''))) ? ' selected' : '';
                $html .= '<option value="' . h($optValue) . '"' . $selected . '>' . h($optLabel) . '</option>';
            }
            $html .= '</select>';
        } elseif ($field['name'] === 'required' || ($field['type'] ?? '') === 'checkbox') {
            // Checkbox pour les valeurs booléennes
            $checked = ($fieldValue === true || $fieldValue === '1' || $fieldValue === 'true' || $fieldValue === 'on') ? ' checked' : '';
            $html .= '<div class="form-check">';
            $html .= '<input type="checkbox" name="' . h($fieldName) . '" class="form-check-input" value="1"' . $checked . ' />';
            $html .= '<label class="form-check-label small">Oui</label>';
            $html .= '</div>';
        } else {
            $html .= '<input type="' . ($field['type'] ?? 'text') . '" name="' . h($fieldName) . '" class="form-control form-control-sm" value="' . h($fieldValue) . '" placeholder="' . h($field['placeholder'] ?? '') . '" />';
        }
        $html .= '</div>';
    }
    $html .= '</div>';
    $html .= '</div>';
    $html .= '</div>';
    return $html;
}

?><!doctype html>
<html lang="fr">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>Administration du contenu – Campagne BIZIMANA Aboubakar</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet" />
  <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet" />
  <style>
    body { background: #f4f6f8; padding: 1.5rem 0; font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif; }
    .bg-brand { background: #0b3d2e; }
    .btn-brand { background: #0b3d2e; color: #fff; border: none; }
    .btn-brand:hover { filter: brightness(1.08); }
    .section-card { margin-bottom: 1rem; border-left: 3px solid #0b3d2e; }
    .section-header { cursor: pointer; user-select: none; transition: background 0.2s; }
    .section-header:hover { background: rgba(11, 61, 46, 0.05); }
    .form-label { font-weight: 600; font-size: 0.9rem; margin-bottom: 0.4rem; color: #495057; }
    .form-label i { margin-right: 0.25rem; color: #0b3d2e; }
    .sticky-top-form { position: sticky; top: 1rem; z-index: 100; }
    .array-item-card { border: 1px solid #dee2e6; background: #fff; }
    .array-item-card:hover { border-color: #0b3d2e; box-shadow: 0 2px 4px rgba(11, 61, 46, 0.1); }
    .array-items-container .array-item-card:last-child { margin-bottom: 0.5rem; }
    .form-control-sm, .form-select-sm { font-size: 0.875rem; }
    @media (max-width: 768px) {
      .section-card .card-header { padding: 0.75rem; }
    }
  </style>
</head>
<body>
  <div class="container">
    <div class="row justify-content-center">
      <div class="col-lg-11">
        <div class="card shadow-sm border-0">
          <div class="card-header bg-brand text-white">
            <div class="d-flex justify-content-between align-items-center">
              <div>
                <h1 class="h5 mb-1"><i class="bi bi-gear"></i> Administration du contenu</h1>
                <small class="d-block">Modifiez les sections ci-dessous. Cliquez sur un en-tête pour plier/déplier.</small>
              </div>
              <a href="index.php" class="btn btn-sm btn-light" target="_blank"><i class="bi bi-eye"></i> Prévisualiser</a>
            </div>
          </div>
          <div class="card-body">
            <?php if ($message): ?>
              <div class="alert alert-success alert-dismissible fade show" role="alert">
                <i class="bi bi-check-circle me-2"></i><?= h($message) ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
              </div>
            <?php endif; ?>
            <?php if (!empty($errors)): ?>
              <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <i class="bi bi-exclamation-triangle me-2"></i><strong>Erreurs détectées :</strong>
                <ul class="mb-0 mt-2">
                  <?php foreach ($errors as $err): ?><li><?= h($err) ?></li><?php endforeach; ?>
                </ul>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
              </div>
            <?php endif; ?>

            <form method="post" class="needs-validation" novalidate id="adminForm">
              <!-- Informations générales -->
              <div class="section-card card mb-3">
                <div class="card-header section-header d-flex justify-content-between align-items-center" data-bs-toggle="collapse" data-bs-target="#section-site">
                  <span><i class="bi bi-info-circle me-2"></i>Informations générales</span>
                  <i class="bi bi-chevron-down"></i>
                </div>
                <div id="section-site" class="collapse show">
                  <div class="card-body row g-3">
                  <div class="col-md-6"><label class="form-label" for="site_title">Titre du site</label><input type="text" id="site_title" name="site_title" class="form-control" value="<?= h(field('site_title', value_from($content, ['site', 'title']), $usePostValues)) ?>" /></div>
                  <div class="col-md-6"><label class="form-label" for="site_theme_color">Couleur de thème</label><input type="text" id="site_theme_color" name="site_theme_color" class="form-control" value="<?= h(field('site_theme_color', value_from($content, ['site', 'theme_color']), $usePostValues)) ?>" /></div>
                  <div class="col-12"><label class="form-label" for="site_meta_description">Meta description</label><textarea id="site_meta_description" name="site_meta_description" class="form-control" rows="2"><?= h(field('site_meta_description', value_from($content, ['site', 'meta_description']), $usePostValues)) ?></textarea></div>
                  <div class="col-md-3"><label class="form-label" for="site_og_type">OG Type</label><input type="text" id="site_og_type" name="site_og_type" class="form-control" value="<?= h(field('site_og_type', value_from($content, ['site','og','type']), $usePostValues)) ?>" /></div>
                  <div class="col-md-3"><label class="form-label" for="site_og_title">OG Title</label><input type="text" id="site_og_title" name="site_og_title" class="form-control" value="<?= h(field('site_og_title', value_from($content, ['site','og','title']), $usePostValues)) ?>" /></div>
                  <div class="col-md-3"><label class="form-label" for="site_og_image">OG Image</label><input type="text" id="site_og_image" name="site_og_image" class="form-control" value="<?= h(field('site_og_image', value_from($content, ['site','og','image']), $usePostValues)) ?>" /></div>
                  <div class="col-md-3"><label class="form-label" for="site_og_url">OG URL</label><input type="text" id="site_og_url" name="site_og_url" class="form-control" value="<?= h(field('site_og_url', value_from($content, ['site','og','url']), $usePostValues)) ?>" /></div>
                  <div class="col-12"><label class="form-label" for="site_og_description">OG Description</label><textarea id="site_og_description" name="site_og_description" class="form-control" rows="2"><?= h(field('site_og_description', value_from($content, ['site','og','description']), $usePostValues)) ?></textarea></div>
                  </div>
                </div>
              </div>

              <!-- Navigation -->
              <div class="section-card card mb-3">
                <div class="card-header section-header d-flex justify-content-between align-items-center" data-bs-toggle="collapse" data-bs-target="#section-nav">
                  <span><i class="bi bi-list-ul me-2"></i>Navigation</span>
                  <i class="bi bi-chevron-down"></i>
                </div>
                <div id="section-nav" class="collapse show">
                  <div class="card-body row g-3">
                  <div class="col-md-6"><label class="form-label" for="nav_brand_label">Nom de marque</label><input type="text" id="nav_brand_label" name="nav_brand_label" class="form-control" value="<?= h(field('nav_brand_label', value_from($content, ['navigation','brand_label']), $usePostValues)) ?>" /></div>
                  <div class="col-md-6"><label class="form-label" for="nav_brand_aria">Texte accessibilité</label><input type="text" id="nav_brand_aria" name="nav_brand_aria" class="form-control" value="<?= h(field('nav_brand_aria', value_from($content, ['navigation','brand_aria']), $usePostValues)) ?>" /></div>
                  <div class="col-12"><label class="form-label" for="nav_cta_label">CTA inscription (label)</label><input type="text" id="nav_cta_label" name="nav_cta_label" class="form-control" value="<?= h(field('nav_cta_label', value_from($content, ['navigation','cta','label']), $usePostValues)) ?>" /></div>
                  <div class="col-12"><label class="form-label" for="nav_cta_href">CTA inscription (URL)</label><input type="text" id="nav_cta_href" name="nav_cta_href" class="form-control" value="<?= h(field('nav_cta_href', value_from($content, ['navigation','cta','href']), $usePostValues)) ?>" /></div>
                  <div class="col-12">
                    <label class="form-label"><i class="bi bi-list-ul"></i>Liens de navigation</label>
                    <?php
                    $navItems = value_from($content, ['navigation', 'items'], []);
                    $navFields = [
                        ['name' => 'label', 'label' => 'Libellé', 'type' => 'text', 'placeholder' => 'Ex: Accueil', 'col' => 'col-md-6'],
                        ['name' => 'href', 'label' => 'Lien (URL)', 'type' => 'text', 'placeholder' => 'Ex: #hero ou /page', 'col' => 'col-md-6'],
                    ];
                    echo render_array_items('nav_item', $navItems, $navFields, 'Lien de navigation');
                    ?>
                  </div>
                  </div>
                </div>
              </div>

              <!-- Hero -->
              <div class="section-card card mb-3">
                <div class="card-header section-header d-flex justify-content-between align-items-center" data-bs-toggle="collapse" data-bs-target="#section-hero">
                  <span><i class="bi bi-star me-2"></i>Hero / Section principale</span>
                  <i class="bi bi-chevron-down"></i>
                </div>
                <div id="section-hero" class="collapse show">
                  <div class="card-body row g-3">
                    <div class="col-md-4"><label class="form-label" for="hero_chip">Badge</label><input type="text" id="hero_chip" name="hero_chip" class="form-control" value="<?= h(field('hero_chip', value_from($content, ['hero','chip']), $usePostValues)) ?>" /></div>
                    <div class="col-12"><label class="form-label" for="hero_title"><i class="bi bi-type"></i>Titre principal</label><textarea id="hero_title" name="hero_title" class="form-control" rows="2"><?= h(field('hero_title', value_from($content, ['hero','title']), $usePostValues)) ?></textarea><div class="field-help">HTML autorisé (ex: &lt;br&gt;, &lt;span&gt;)</div></div>
                    <div class="col-12"><label class="form-label" for="hero_subtitle">Sous-titre</label><textarea id="hero_subtitle" name="hero_subtitle" class="form-control" rows="2"><?= h(field('hero_subtitle', value_from($content, ['hero','subtitle']), $usePostValues)) ?></textarea></div>
                    <div class="col-md-6"><label class="form-label" for="hero_share_label">Label partage</label><input type="text" id="hero_share_label" name="hero_share_label" class="form-control" value="<?= h(field('hero_share_label', value_from($content, ['hero','share_label']), $usePostValues)) ?>" /></div>
                    <div class="col-md-6"><label class="form-label" for="hero_share_prefix">Message WhatsApp</label><input type="text" id="hero_share_prefix" name="hero_share_prefix" class="form-control" value="<?= h(field('hero_share_prefix', value_from($content, ['hero','share_message_prefix']), $usePostValues)) ?>" /></div>
                    <div class="col-md-4"><label class="form-label" for="hero_about_subtitle">Sous-titre carte</label><input type="text" id="hero_about_subtitle" name="hero_about_subtitle" class="form-control" value="<?= h(field('hero_about_subtitle', value_from($content, ['hero','about','subtitle']), $usePostValues)) ?>" /></div>
                    <div class="col-md-4"><label class="form-label" for="hero_about_title">Titre carte</label><input type="text" id="hero_about_title" name="hero_about_title" class="form-control" value="<?= h(field('hero_about_title', value_from($content, ['hero','about','title']), $usePostValues)) ?>" /></div>
                    <div class="col-12"><label class="form-label" for="hero_about_description">Description carte</label><textarea id="hero_about_description" name="hero_about_description" class="form-control" rows="2"><?= h(field('hero_about_description', value_from($content, ['hero','about','description']), $usePostValues)) ?></textarea></div>
                    <div class="col-12">
                      <label class="form-label"><i class="bi bi-bar-chart"></i>Statistiques</label>
                      <?php
                      $heroStats = value_from($content, ['hero', 'stats'], []);
                      $statsFields = [
                          ['name' => 'value', 'label' => 'Valeur', 'type' => 'text', 'placeholder' => 'Ex: 20+', 'col' => 'col-md-4'],
                          ['name' => 'label', 'label' => 'Libellé', 'type' => 'text', 'placeholder' => 'Ex: Années de service', 'col' => 'col-md-8'],
                      ];
                      echo render_array_items('hero_stat', $heroStats, $statsFields, 'Statistique');
                      ?>
                    </div>
                    <div class="col-12">
                      <label class="form-label"><i class="bi bi-cursor"></i>Boutons d'action</label>
                      <?php
                      $heroCtas = value_from($content, ['hero', 'ctas'], []);
                      $ctaFields = [
                          ['name' => 'style', 'label' => 'Style', 'type' => 'select', 'options' => ['primary' => 'Primary', 'secondary' => 'Secondary', 'outline-primary' => 'Outline Primary'], 'col' => 'col-md-3'],
                          ['name' => 'label', 'label' => 'Libellé', 'type' => 'text', 'placeholder' => 'Texte du bouton', 'col' => 'col-md-4'],
                          ['name' => 'href', 'label' => 'Lien', 'type' => 'text', 'placeholder' => '#section ou /page', 'col' => 'col-md-3'],
                          ['name' => 'icon', 'label' => 'Icône', 'type' => 'text', 'placeholder' => 'bi bi-icon', 'col' => 'col-md-2'],
                      ];
                      echo render_array_items('hero_cta', $heroCtas, $ctaFields, 'Bouton');
                      ?>
                    </div>
                    <div class="col-12">
                      <label class="form-label"><i class="bi bi-star"></i>Points forts</label>
                      <?php
                      $heroHighlights = value_from($content, ['hero', 'about', 'highlights'], []);
                      // Convertir les strings en objets pour le formulaire
                      $highlightsData = [];
                      foreach ($heroHighlights as $hl) {
                        if (is_string($hl)) {
                          $highlightsData[] = ['text' => $hl, 'icon' => ''];
                        } else {
                          $highlightsData[] = ['text' => $hl['text'] ?? '', 'icon' => $hl['icon'] ?? ''];
                        }
                      }
                      $highlightFields = [
                          ['name' => 'text', 'label' => 'Texte', 'type' => 'text', 'placeholder' => 'Description du point fort', 'col' => 'col-md-10'],
                          ['name' => 'icon', 'label' => 'Icône (optionnel)', 'type' => 'text', 'placeholder' => 'bi bi-check', 'col' => 'col-md-2'],
                      ];
                      echo render_array_items('hero_highlight', $highlightsData, $highlightFields, 'Point fort');
                      ?>
                    </div>
                    <div class="col-md-4"><label class="form-label" for="hero_about_cta_label">CTA (label)</label><input type="text" id="hero_about_cta_label" name="hero_about_cta_label" class="form-control" value="<?= h(field('hero_about_cta_label', value_from($content, ['hero','about','cta','label']), $usePostValues)) ?>" /></div>
                    <div class="col-md-4"><label class="form-label" for="hero_about_cta_href">CTA (lien)</label><input type="text" id="hero_about_cta_href" name="hero_about_cta_href" class="form-control" value="<?= h(field('hero_about_cta_href', value_from($content, ['hero','about','cta','href']), $usePostValues)) ?>" /></div>
                    <div class="col-md-4"><label class="form-label" for="hero_about_cta_icon">CTA (icône Bootstrap)</label><input type="text" id="hero_about_cta_icon" name="hero_about_cta_icon" class="form-control" value="<?= h(field('hero_about_cta_icon', value_from($content, ['hero','about','cta','icon']), $usePostValues)) ?>" placeholder="bi bi-person-heart" /></div>
                  </div>
                </div>
              </div>

              <!-- Portrait -->
              <div class="section-card card mb-3">
                <div class="card-header section-header d-flex justify-content-between align-items-center" data-bs-toggle="collapse" data-bs-target="#section-portrait">
                  <span><i class="bi bi-person me-2"></i>Portrait du candidat</span>
                  <i class="bi bi-chevron-down"></i>
                </div>
                <div id="section-portrait" class="collapse">
                  <div class="card-body row g-3">
                  <div class="col-md-6"><label class="form-label" for="portrait_title">Titre</label><input type="text" id="portrait_title" name="portrait_title" class="form-control" value="<?= h(field('portrait_title', value_from($content, ['portrait','title']), $usePostValues)) ?>" /></div>
                  <div class="col-md-6"><label class="form-label" for="portrait_image_alt">Texte alternatif photo</label><input type="text" id="portrait_image_alt" name="portrait_image_alt" class="form-control" value="<?= h(field('portrait_image_alt', value_from($content, ['portrait','image_alt']), $usePostValues)) ?>" /></div>
                  <div class="col-12"><label class="form-label" for="portrait_description">Description</label><textarea id="portrait_description" name="portrait_description" class="form-control" rows="2"><?= h(field('portrait_description', value_from($content, ['portrait','description']), $usePostValues)) ?></textarea></div>
                  <div class="col-12">
                    <label class="form-label"><i class="bi bi-trophy"></i>Réalisations</label>
                    <?php
                    $portraitAchievements = value_from($content, ['portrait', 'achievements'], []);
                    $achievementFields = [
                        ['name' => 'icon', 'label' => 'Icône', 'type' => 'text', 'placeholder' => 'bi bi-award', 'col' => 'col-md-3'],
                        ['name' => 'text', 'label' => 'Description', 'type' => 'textarea', 'rows' => 2, 'placeholder' => 'Description de la réalisation', 'col' => 'col-md-9'],
                    ];
                    echo render_array_items('portrait_achievement', $portraitAchievements, $achievementFields, 'Réalisation');
                    ?>
                  </div>
                  <div class="col-12">
                    <label class="form-label"><i class="bi bi-bar-chart-line"></i>Statistiques</label>
                    <?php
                    $portraitStats = value_from($content, ['portrait', 'stats'], []);
                    $portraitStatFields = [
                        ['name' => 'value', 'label' => 'Valeur', 'type' => 'text', 'placeholder' => 'Ex: 3', 'col' => 'col-md-4'],
                        ['name' => 'label', 'label' => 'Libellé', 'type' => 'text', 'placeholder' => 'Ex: Programmes nationaux', 'col' => 'col-md-8'],
                    ];
                    echo render_array_items('portrait_stat', $portraitStats, $portraitStatFields, 'Statistique');
                    ?>
                  </div>
                  <div class="col-md-6"><label class="form-label" for="portrait_convictions_heading">Titre convictions</label><input type="text" id="portrait_convictions_heading" name="portrait_convictions_heading" class="form-control" value="<?= h(field('portrait_convictions_heading', value_from($content, ['portrait','convictions','heading']), $usePostValues)) ?>" /></div>
                  <div class="col-12">
                    <label class="form-label"><i class="bi bi-heart"></i>Convictions</label>
                    <?php
                    $portraitConvictions = value_from($content, ['portrait', 'convictions', 'items'], []);
                    $convictionFields = [
                        ['name' => 'title', 'label' => 'Titre', 'type' => 'text', 'placeholder' => 'Ex: Service', 'col' => 'col-md-4'],
                        ['name' => 'text', 'label' => 'Description', 'type' => 'textarea', 'rows' => 2, 'placeholder' => 'Description de la conviction', 'col' => 'col-md-8'],
                    ];
                    echo render_array_items('portrait_conviction', $portraitConvictions, $convictionFields, 'Conviction');
                    ?>
                  </div>
                  </div>
                </div>
              </div>

              <!-- D'autres sections suivent la même structure -->
              <div class="section-card card mb-3">
                <div class="card-header">Projet</div>
                <div class="card-body row g-3">
                  <div class="col-md-4"><label class="form-label" for="project_eyebrow">Surtitre</label><input type="text" id="project_eyebrow" name="project_eyebrow" class="form-control" value="<?= h(field('project_eyebrow', value_from($content, ['project','eyebrow']), $usePostValues)) ?>" /></div>
                  <div class="col-md-8"><label class="form-label" for="project_title">Titre</label><input type="text" id="project_title" name="project_title" class="form-control" value="<?= h(field('project_title', value_from($content, ['project','title']), $usePostValues)) ?>" /></div>
                  <div class="col-12"><label class="form-label" for="project_description">Description</label><textarea id="project_description" name="project_description" class="form-control" rows="2"><?= h(field('project_description', value_from($content, ['project','description']), $usePostValues)) ?></textarea></div>
                  <div class="col-12">
                    <label class="form-label"><i class="bi bi-hourglass-split"></i>Segments</label>
                    <?php
                    $projectSegments = value_from($content, ['project', 'segments'], []);
                    $segmentFields = [
                        ['name' => 'title', 'label' => 'Titre', 'type' => 'text', 'placeholder' => 'Ex: Accélérateur 100 jours', 'col' => 'col-md-4'],
                        ['name' => 'text', 'label' => 'Description', 'type' => 'textarea', 'rows' => 2, 'placeholder' => 'Description du segment', 'col' => 'col-md-8'],
                    ];
                    echo render_array_items('project_segment', $projectSegments, $segmentFields, 'Segment');
                    ?>
                  </div>
                  <div class="col-12">
                    <label class="form-label"><i class="bi bi-cursor"></i>Boutons d'action</label>
                    <?php
                    $projectCtas = value_from($content, ['project', 'ctas'], []);
                    $projectCtaFields = [
                        ['name' => 'style', 'label' => 'Style', 'type' => 'select', 'options' => ['primary' => 'Primary', 'secondary' => 'Secondary', 'outline-primary' => 'Outline Primary'], 'col' => 'col-md-3'],
                        ['name' => 'label', 'label' => 'Libellé', 'type' => 'text', 'placeholder' => 'Texte du bouton', 'col' => 'col-md-4'],
                        ['name' => 'href', 'label' => 'Lien', 'type' => 'text', 'placeholder' => '#section ou /page', 'col' => 'col-md-3'],
                        ['name' => 'icon', 'label' => 'Icône', 'type' => 'text', 'placeholder' => 'bi bi-icon', 'col' => 'col-md-2'],
                    ];
                    echo render_array_items('project_cta', $projectCtas, $projectCtaFields, 'Bouton');
                    ?>
                  </div>
                  <div class="col-12">
                    <label class="form-label"><i class="bi bi-grid-3x3"></i>Cartes</label>
                    <?php
                    $projectCards = value_from($content, ['project', 'cards'], []);
                    $projectCardFields = [
                        ['name' => 'icon', 'label' => 'Icône', 'type' => 'text', 'placeholder' => 'bi bi-broadcast', 'col' => 'col-md-3'],
                        ['name' => 'title', 'label' => 'Titre', 'type' => 'text', 'placeholder' => 'Titre de la carte', 'col' => 'col-md-9'],
                        ['name' => 'text', 'label' => 'Description', 'type' => 'textarea', 'rows' => 2, 'placeholder' => 'Description de la carte', 'col' => 'col-12'],
                    ];
                    echo render_array_items('project_card', $projectCards, $projectCardFields, 'Carte');
                    ?>
                  </div>
                </div>
              </div>

              <div class="section-card card mb-3">
                <div class="card-header">Vision</div>
                <div class="card-body row g-3">
                  <div class="col-md-4"><label class="form-label" for="vision_eyebrow">Surtitre</label><input type="text" id="vision_eyebrow" name="vision_eyebrow" class="form-control" value="<?= h(field('vision_eyebrow', value_from($content, ['vision','eyebrow']), $usePostValues)) ?>" /></div>
                  <div class="col-md-8"><label class="form-label" for="vision_title">Titre</label><input type="text" id="vision_title" name="vision_title" class="form-control" value="<?= h(field('vision_title', value_from($content, ['vision','title']), $usePostValues)) ?>" /></div>
                  <div class="col-12"><label class="form-label" for="vision_subtitle">Sous-titre</label><textarea id="vision_subtitle" name="vision_subtitle" class="form-control" rows="2"><?= h(field('vision_subtitle', value_from($content, ['vision','subtitle']), $usePostValues)) ?></textarea></div>
                  <div class="col-md-6"><label class="form-label" for="vision_objective_title">Objectif (titre)</label><input type="text" id="vision_objective_title" name="vision_objective_title" class="form-control" value="<?= h(field('vision_objective_title', value_from($content, ['vision','objective','title']), $usePostValues)) ?>" /></div>
                  <div class="col-md-6"><label class="form-label" for="vision_objective_text">Objectif (texte)</label><textarea id="vision_objective_text" name="vision_objective_text" class="form-control" rows="2"><?= h(field('vision_objective_text', value_from($content, ['vision','objective','text']), $usePostValues)) ?></textarea></div>
                  <div class="col-12">
                    <label class="form-label"><i class="bi bi-grid-3x3"></i>Cartes Vision/Mission</label>
                    <?php
                    $visionCards = value_from($content, ['vision', 'cards'], []);
                    $visionCardFields = [
                        ['name' => 'title', 'label' => 'Titre', 'type' => 'text', 'placeholder' => 'Ex: Vision', 'col' => 'col-md-4'],
                        ['name' => 'text', 'label' => 'Description', 'type' => 'textarea', 'rows' => 2, 'placeholder' => 'Description', 'col' => 'col-md-8'],
                    ];
                    echo render_array_items('vision_card', $visionCards, $visionCardFields, 'Carte');
                    ?>
                  </div>
                  <div class="col-12">
                    <label class="form-label"><i class="bi bi-list-check"></i>Objectifs spécifiques</label>
                    <?php
                    $visionSpecific = value_from($content, ['vision', 'specific_items'], []);
                    $visionSpecificFields = [
                        ['name' => 'title', 'label' => 'Titre', 'type' => 'text', 'placeholder' => 'Ex: Gouvernance & transparence', 'col' => 'col-md-4'],
                        ['name' => 'text', 'label' => 'Description', 'type' => 'textarea', 'rows' => 2, 'placeholder' => 'Description de l\'objectif', 'col' => 'col-md-8'],
                    ];
                    echo render_array_items('vision_specific', $visionSpecific, $visionSpecificFields, 'Objectif');
                    ?>
                  </div>
                  <div class="col-12"><label class="form-label" for="vision_specific_title">Titre bloc objectifs</label><input type="text" id="vision_specific_title" name="vision_specific_title" class="form-control" value="<?= h(field('vision_specific_title', value_from($content, ['vision','specific_title']), $usePostValues)) ?>" /></div>
                </div>
              </div>

              <!-- Sections supplémentaires : piliers, activités, manifeste, agenda, médias, témoignages, FAQ, contact, footer -->
              <div class="section-card card mb-3">
                <div class="card-header">Piliers</div>
                <div class="card-body row g-3">
                  <div class="col-md-4"><label class="form-label" for="pillars_eyebrow">Surtitre</label><input type="text" id="pillars_eyebrow" name="pillars_eyebrow" class="form-control" value="<?= h(field('pillars_eyebrow', value_from($content, ['pillars','eyebrow']), $usePostValues)) ?>" /></div>
                  <div class="col-md-8"><label class="form-label" for="pillars_title">Titre</label><input type="text" id="pillars_title" name="pillars_title" class="form-control" value="<?= h(field('pillars_title', value_from($content, ['pillars','title']), $usePostValues)) ?>" /></div>
                  <div class="col-12"><label class="form-label" for="pillars_subtitle">Sous-titre</label><textarea id="pillars_subtitle" name="pillars_subtitle" class="form-control" rows="2"><?= h(field('pillars_subtitle', value_from($content, ['pillars','subtitle']), $usePostValues)) ?></textarea></div>
                  <div class="col-12">
                    <label class="form-label"><i class="bi bi-columns"></i>Piliers</label>
                    <?php
                    $pillarsItems = value_from($content, ['pillars', 'items'], []);
                    $pillarFields = [
                        ['name' => 'icon', 'label' => 'Icône', 'type' => 'text', 'placeholder' => 'bi bi-heart-pulse', 'col' => 'col-md-3'],
                        ['name' => 'title', 'label' => 'Titre', 'type' => 'text', 'placeholder' => 'Ex: Servir avec cœur', 'col' => 'col-md-4'],
                        ['name' => 'text', 'label' => 'Description', 'type' => 'textarea', 'rows' => 2, 'placeholder' => 'Description du pilier', 'col' => 'col-md-5'],
                    ];
                    echo render_array_items('pillar', $pillarsItems, $pillarFields, 'Pilier');
                    ?>
                  </div>
                </div>
              </div>

              <div class="section-card card mb-3">
                <div class="card-header">Activités</div>
                <div class="card-body row g-3">
                  <div class="col-md-4"><label class="form-label" for="activities_eyebrow">Surtitre</label><input type="text" id="activities_eyebrow" name="activities_eyebrow" class="form-control" value="<?= h(field('activities_eyebrow', value_from($content, ['activities','eyebrow']), $usePostValues)) ?>" /></div>
                  <div class="col-md-8"><label class="form-label" for="activities_title">Titre</label><input type="text" id="activities_title" name="activities_title" class="form-control" value="<?= h(field('activities_title', value_from($content, ['activities','title']), $usePostValues)) ?>" /></div>
                  <div class="col-12"><label class="form-label" for="activities_subtitle">Sous-titre</label><textarea id="activities_subtitle" name="activities_subtitle" class="form-control" rows="2"><?= h(field('activities_subtitle', value_from($content, ['activities','subtitle']), $usePostValues)) ?></textarea></div>
                  <div class="col-12">
                    <label class="form-label"><i class="bi bi-list-task"></i>Blocs d'activités</label>
                    <div class="alert alert-info small mb-3">
                      <i class="bi bi-info-circle"></i> Les blocs d'activités permettent d'organiser les activités par catégories.
                    </div>
                    <?php
                    $activitiesBlocks = value_from($content, ['activities', 'blocks'], []);
                    // Pour les blocs d'activités, on a besoin d'une structure spéciale avec items
                    $blockIndex = 0;
                    foreach ($activitiesBlocks as $block) {
                      echo '<div class="activity-block-card card mb-3" data-index="' . $blockIndex . '">';
                      echo '<div class="card-body">';
                      echo '<div class="d-flex justify-content-between align-items-center mb-2">';
                      echo '<small class="text-muted fw-bold">Bloc #' . ($blockIndex + 1) . '</small>';
                      echo '<button type="button" class="btn btn-sm btn-outline-danger" onclick="removeActivityBlock(this)"><i class="bi bi-trash"></i> Supprimer le bloc</button>';
                      echo '</div>';
                      echo '<div class="row g-2 mb-3">';
                      echo '<div class="col-md-8"><label class="form-label small">Titre du bloc</label>';
                      echo '<input type="text" name="activity_block_' . $blockIndex . '_title" class="form-control form-control-sm" value="' . h($block['title'] ?? '') . '" placeholder="Ex: A. Pour le bon goût de l\'esprit scout" /></div>';
                      echo '<div class="col-md-4"><label class="form-label small">Largeur colonne</label>';
                      echo '<select name="activity_block_' . $blockIndex . '_col" class="form-select form-select-sm">';
                      echo '<option value="col-md-12"' . (($block['col'] ?? 'col-md-12') === 'col-md-12' ? ' selected' : '') . '>Pleine largeur</option>';
                      echo '<option value="col-md-6"' . (($block['col'] ?? '') === 'col-md-6' ? ' selected' : '') . '>Demi largeur</option>';
                      echo '<option value="col-md-4"' . (($block['col'] ?? '') === 'col-md-4' ? ' selected' : '') . '>Tiers largeur</option>';
                      echo '</select></div>';
                      echo '</div>';
                      // Items du bloc
                      $items = $block['items'] ?? [];
                      $itemIndex = 0;
                      echo '<div class="activity-block-items" data-block-index="' . $blockIndex . '">';
                      foreach ($items as $item) {
                        echo '<div class="d-flex mb-2 align-items-start">';
                        echo '<textarea name="activity_block_' . $blockIndex . '_item_' . $itemIndex . '" class="form-control form-control-sm me-2" rows="2" placeholder="Item d\'activité (HTML autorisé)">' . h(is_string($item) ? $item : '') . '</textarea>';
                        echo '<button type="button" class="btn btn-sm btn-outline-danger" onclick="removeActivityItem(this)"><i class="bi bi-x"></i></button>';
                        echo '</div>';
                        $itemIndex++;
                      }
                      echo '</div>';
                      echo '<button type="button" class="btn btn-sm btn-outline-primary" onclick="addActivityItem(' . $blockIndex . ')"><i class="bi bi-plus-circle"></i> Ajouter un item</button>';
                      echo '</div></div>';
                      $blockIndex++;
                    }
                    ?>
                    <button type="button" class="btn btn-sm btn-outline-primary mt-2" onclick="addActivityBlock()"><i class="bi bi-plus-circle"></i> Ajouter un bloc d'activités</button>
                  </div>
                </div>
              </div>

              <div class="section-card card mb-3">
                <div class="card-header">Manifeste</div>
                <div class="card-body row g-3">
                  <div class="col-md-4"><label class="form-label" for="manifeste_eyebrow">Surtitre</label><input type="text" id="manifeste_eyebrow" name="manifeste_eyebrow" class="form-control" value="<?= h(field('manifeste_eyebrow', value_from($content, ['manifeste','eyebrow']), $usePostValues)) ?>" /></div>
                  <div class="col-md-8"><label class="form-label" for="manifeste_title">Titre</label><input type="text" id="manifeste_title" name="manifeste_title" class="form-control" value="<?= h(field('manifeste_title', value_from($content, ['manifeste','title']), $usePostValues)) ?>" /></div>
                  <div class="col-md-4"><label class="form-label" for="manifeste_image_alt">Texte alternatif image</label><input type="text" id="manifeste_image_alt" name="manifeste_image_alt" class="form-control" value="<?= h(field('manifeste_image_alt', value_from($content, ['manifeste','image_alt']), $usePostValues)) ?>" /></div>
                  <div class="col-12">
                    <label class="form-label"><i class="bi bi-cursor"></i>Boutons d'action</label>
                    <?php
                    $manifesteCtas = value_from($content, ['manifeste', 'ctas'], []);
                    $manifesteCtaFields = [
                        ['name' => 'style', 'label' => 'Style', 'type' => 'select', 'options' => ['primary' => 'Primary', 'secondary' => 'Secondary', 'outline-primary' => 'Outline Primary'], 'col' => 'col-md-3'],
                        ['name' => 'label', 'label' => 'Libellé', 'type' => 'text', 'placeholder' => 'Texte du bouton', 'col' => 'col-md-4'],
                        ['name' => 'href', 'label' => 'Lien', 'type' => 'text', 'placeholder' => '#section ou /page', 'col' => 'col-md-3'],
                        ['name' => 'icon', 'label' => 'Icône', 'type' => 'text', 'placeholder' => 'bi bi-icon', 'col' => 'col-md-2'],
                    ];
                    echo render_array_items('manifeste_cta', $manifesteCtas, $manifesteCtaFields, 'Bouton');
                    ?>
                  </div>
                  <div class="col-12"><label class="form-label" for="manifeste_description">Description</label><textarea id="manifeste_description" name="manifeste_description" class="form-control" rows="2"><?= h(field('manifeste_description', value_from($content, ['manifeste','description']), $usePostValues)) ?></textarea></div>
                </div>
              </div>

              <div class="section-card card mb-3">
                <div class="card-header">Agenda</div>
                <div class="card-body row g-3">
                  <div class="col-md-4"><label class="form-label" for="agenda_eyebrow">Surtitre</label><input type="text" id="agenda_eyebrow" name="agenda_eyebrow" class="form-control" value="<?= h(field('agenda_eyebrow', value_from($content, ['agenda','eyebrow']), $usePostValues)) ?>" /></div>
                  <div class="col-md-8"><label class="form-label" for="agenda_title">Titre</label><input type="text" id="agenda_title" name="agenda_title" class="form-control" value="<?= h(field('agenda_title', value_from($content, ['agenda','title']), $usePostValues)) ?>" /></div>
                  <div class="col-12">
                    <label class="form-label"><i class="bi bi-clock-history"></i>Frise chronologique</label>
                    <?php
                    $agendaTimeline = value_from($content, ['agenda', 'timeline'], []);
                    $timelineFields = [
                        ['name' => 'title', 'label' => 'Titre', 'type' => 'text', 'placeholder' => 'Ex: Semaine 1 – Lancement', 'col' => 'col-md-4'],
                        ['name' => 'text', 'label' => 'Description', 'type' => 'textarea', 'rows' => 2, 'placeholder' => 'Description de l\'étape', 'col' => 'col-md-8'],
                    ];
                    echo render_array_items('agenda_timeline', $agendaTimeline, $timelineFields, 'Étape');
                    ?>
                  </div>
                  <div class="col-md-4"><label class="form-label" for="agenda_form_heading">Titre formulaire</label><input type="text" id="agenda_form_heading" name="agenda_form_heading" class="form-control" value="<?= h(field('agenda_form_heading', value_from($content, ['agenda','form','heading']), $usePostValues)) ?>" /></div>
                  <div class="col-md-4"><label class="form-label" for="agenda_form_submit">Label soumission</label><input type="text" id="agenda_form_submit" name="agenda_form_submit" class="form-control" value="<?= h(field('agenda_form_submit', value_from($content, ['agenda','form','submit_label']), $usePostValues)) ?>" /></div>
                  <div class="col-12">
                    <label class="form-label"><i class="bi bi-input-cursor-text"></i>Champs du formulaire</label>
                    <?php
                    $agendaFields = value_from($content, ['agenda', 'form', 'fields'], []);
                    // Note: Les champs de formulaire peuvent avoir une structure complexe, on simplifie ici
                    $agendaFieldFields = [
                        ['name' => 'id', 'label' => 'ID', 'type' => 'text', 'placeholder' => 'id_du_champ', 'col' => 'col-md-2'],
                        ['name' => 'name', 'label' => 'Nom', 'type' => 'text', 'placeholder' => 'name', 'col' => 'col-md-2'],
                        ['name' => 'label', 'label' => 'Libellé', 'type' => 'text', 'placeholder' => 'Label du champ', 'col' => 'col-md-3'],
                        ['name' => 'type', 'label' => 'Type', 'type' => 'select', 'options' => ['text' => 'Texte', 'email' => 'Email', 'textarea' => 'Zone de texte', 'select' => 'Liste'], 'col' => 'col-md-2'],
                        ['name' => 'placeholder', 'label' => 'Placeholder', 'type' => 'text', 'placeholder' => 'Texte indicatif', 'col' => 'col-md-2'],
                        ['name' => 'required', 'label' => 'Requis', 'type' => 'checkbox', 'col' => 'col-md-1'],
                    ];
                    echo render_array_items('agenda_field', $agendaFields, $agendaFieldFields, 'Champ');
                    ?>
                  </div>
                  <div class="col-12"><label class="form-label" for="agenda_form_disclaimer">Disclaimer</label><textarea id="agenda_form_disclaimer" name="agenda_form_disclaimer" class="form-control" rows="2"><?= h(field('agenda_form_disclaimer', value_from($content, ['agenda','form','disclaimer']), $usePostValues)) ?></textarea></div>
                </div>
              </div>

              <div class="section-card card mb-3">
                <div class="card-header">Médias</div>
                <div class="card-body row g-3">
                  <div class="col-md-4"><label class="form-label" for="media_eyebrow">Surtitre</label><input type="text" id="media_eyebrow" name="media_eyebrow" class="form-control" value="<?= h(field('media_eyebrow', value_from($content, ['media','eyebrow']), $usePostValues)) ?>" /></div>
                  <div class="col-md-8"><label class="form-label" for="media_title">Titre</label><input type="text" id="media_title" name="media_title" class="form-control" value="<?= h(field('media_title', value_from($content, ['media','title']), $usePostValues)) ?>" /></div>
                  <div class="col-12"><label class="form-label" for="media_subtitle">Sous-titre</label><textarea id="media_subtitle" name="media_subtitle" class="form-control" rows="2"><?= h(field('media_subtitle', value_from($content, ['media','subtitle']), $usePostValues)) ?></textarea></div>
                  <div class="col-12">
                    <label class="form-label"><i class="bi bi-images"></i>Éléments médias</label>
                    <?php
                    $mediaItems = value_from($content, ['media', 'items'], []);
                    $mediaFields = [
                        ['name' => 'image', 'label' => 'Image (URL)', 'type' => 'text', 'placeholder' => '/assets/media1.jpg', 'col' => 'col-md-6'],
                        ['name' => 'alt', 'label' => 'Texte alternatif', 'type' => 'text', 'placeholder' => 'Description de l\'image', 'col' => 'col-md-6'],
                        ['name' => 'title', 'label' => 'Titre', 'type' => 'text', 'placeholder' => 'Titre de l\'élément', 'col' => 'col-md-6'],
                        ['name' => 'text', 'label' => 'Description', 'type' => 'textarea', 'rows' => 2, 'placeholder' => 'Description', 'col' => 'col-md-6'],
                        ['name' => 'href', 'label' => 'Lien', 'type' => 'text', 'placeholder' => '# ou /page', 'col' => 'col-md-4'],
                        ['name' => 'link_label', 'label' => 'Libellé du lien', 'type' => 'text', 'placeholder' => 'Voir plus', 'col' => 'col-md-8'],
                    ];
                    echo render_array_items('media_item', $mediaItems, $mediaFields, 'Élément média');
                    ?>
                  </div>
                </div>
              </div>

              <div class="section-card card mb-3">
                <div class="card-header">Témoignages</div>
                <div class="card-body row g-3">
                  <div class="col-md-4"><label class="form-label" for="testimonials_eyebrow">Surtitre</label><input type="text" id="testimonials_eyebrow" name="testimonials_eyebrow" class="form-control" value="<?= h(field('testimonials_eyebrow', value_from($content, ['testimonials','eyebrow']), $usePostValues)) ?>" /></div>
                  <div class="col-md-8"><label class="form-label" for="testimonials_title">Titre</label><input type="text" id="testimonials_title" name="testimonials_title" class="form-control" value="<?= h(field('testimonials_title', value_from($content, ['testimonials','title']), $usePostValues)) ?>" /></div>
                  <div class="col-12">
                    <label class="form-label"><i class="bi bi-chat-quote"></i>Témoignages</label>
                    <?php
                    $testimonialsItems = value_from($content, ['testimonials', 'items'], []);
                    $testimonialFields = [
                        ['name' => 'quote', 'label' => 'Citation', 'type' => 'textarea', 'rows' => 2, 'placeholder' => 'Le texte du témoignage', 'col' => 'col-md-8'],
                        ['name' => 'author', 'label' => 'Auteur', 'type' => 'text', 'placeholder' => 'Ex: Chef régional, Imbo', 'col' => 'col-md-4'],
                    ];
                    echo render_array_items('testimonial', $testimonialsItems, $testimonialFields, 'Témoignage');
                    ?>
                  </div>
                </div>
              </div>

              <div class="section-card card mb-3">
                <div class="card-header">FAQ</div>
                <div class="card-body row g-3">
                  <div class="col-md-4"><label class="form-label" for="faq_eyebrow">Surtitre</label><input type="text" id="faq_eyebrow" name="faq_eyebrow" class="form-control" value="<?= h(field('faq_eyebrow', value_from($content, ['faq','eyebrow']), $usePostValues)) ?>" /></div>
                  <div class="col-md-8"><label class="form-label" for="faq_title">Titre</label><input type="text" id="faq_title" name="faq_title" class="form-control" value="<?= h(field('faq_title', value_from($content, ['faq','title']), $usePostValues)) ?>" /></div>
                  <div class="col-12">
                    <label class="form-label"><i class="bi bi-question-circle"></i>Questions fréquentes</label>
                    <?php
                    $faqItems = value_from($content, ['faq', 'items'], []);
                    $faqFields = [
                        ['name' => 'question', 'label' => 'Question', 'type' => 'text', 'placeholder' => 'Ex: En quoi cette campagne est-elle différente ?', 'col' => 'col-md-5'],
                        ['name' => 'answer', 'label' => 'Réponse', 'type' => 'textarea', 'rows' => 2, 'placeholder' => 'Réponse détaillée', 'col' => 'col-md-7'],
                    ];
                    echo render_array_items('faq', $faqItems, $faqFields, 'Question/Réponse');
                    ?>
                  </div>
                </div>
              </div>

              <div class="section-card card mb-3">
                <div class="card-header">Contact</div>
                <div class="card-body row g-3">
                  <div class="col-md-4"><label class="form-label" for="contact_eyebrow">Surtitre</label><input type="text" id="contact_eyebrow" name="contact_eyebrow" class="form-control" value="<?= h(field('contact_eyebrow', value_from($content, ['contact','eyebrow']), $usePostValues)) ?>" /></div>
                  <div class="col-md-8"><label class="form-label" for="contact_title">Titre</label><input type="text" id="contact_title" name="contact_title" class="form-control" value="<?= h(field('contact_title', value_from($content, ['contact','title']), $usePostValues)) ?>" /></div>
                  <div class="col-12"><label class="form-label" for="contact_subtitle">Sous-titre</label><textarea id="contact_subtitle" name="contact_subtitle" class="form-control" rows="2"><?= h(field('contact_subtitle', value_from($content, ['contact','subtitle']), $usePostValues)) ?></textarea></div>
                  <div class="col-md-4"><label class="form-label" for="contact_card_title">Titre carte</label><input type="text" id="contact_card_title" name="contact_card_title" class="form-control" value="<?= h(field('contact_card_title', value_from($content, ['contact','contacts_card','title']), $usePostValues)) ?>" /></div>
                  <div class="col-md-4"><label class="form-label" for="contact_social_title">Titre réseaux sociaux</label><input type="text" id="contact_social_title" name="contact_social_title" class="form-control" value="<?= h(field('contact_social_title', value_from($content, ['contact','contacts_card','social_title']), $usePostValues)) ?>" /></div>
                  <div class="col-12">
                    <label class="form-label"><i class="bi bi-telephone"></i>Coordonnées</label>
                    <?php
                    $contactCardItems = value_from($content, ['contact', 'contacts_card', 'items'], []);
                    $contactCardFields = [
                        ['name' => 'icon', 'label' => 'Icône', 'type' => 'text', 'placeholder' => 'bi bi-envelope', 'col' => 'col-md-3'],
                        ['name' => 'text', 'label' => 'Texte', 'type' => 'text', 'placeholder' => 'Ex: contact@example.bi', 'col' => 'col-md-6'],
                        ['name' => 'href', 'label' => 'Lien (optionnel)', 'type' => 'text', 'placeholder' => 'mailto:... ou https://...', 'col' => 'col-md-3'],
                    ];
                    echo render_array_items('contact_card_item', $contactCardItems, $contactCardFields, 'Coordonnée');
                    ?>
                  </div>
                  <div class="col-12">
                    <label class="form-label"><i class="bi bi-share"></i>Réseaux sociaux</label>
                    <?php
                    $contactSocial = value_from($content, ['contact', 'contacts_card', 'social_links'], []);
                    $socialFields = [
                        ['name' => 'icon', 'label' => 'Icône', 'type' => 'text', 'placeholder' => 'bi bi-facebook', 'col' => 'col-md-3'],
                        ['name' => 'href', 'label' => 'URL', 'type' => 'text', 'placeholder' => 'https://facebook.com/...', 'col' => 'col-md-6'],
                        ['name' => 'aria_label', 'label' => 'Label accessibilité', 'type' => 'text', 'placeholder' => 'Facebook', 'col' => 'col-md-3'],
                    ];
                    echo render_array_items('contact_social', $contactSocial, $socialFields, 'Réseau social');
                    ?>
                  </div>
                  <div class="col-12">
                    <label class="form-label"><i class="bi bi-input-cursor-text"></i>Champs du formulaire</label>
                    <?php
                    $contactFormFields = value_from($content, ['contact', 'form', 'fields'], []);
                    $contactFormFieldFields = [
                        ['name' => 'id', 'label' => 'ID', 'type' => 'text', 'placeholder' => 'id_du_champ', 'col' => 'col-md-2'],
                        ['name' => 'name', 'label' => 'Nom', 'type' => 'text', 'placeholder' => 'name', 'col' => 'col-md-2'],
                        ['name' => 'label', 'label' => 'Libellé', 'type' => 'text', 'placeholder' => 'Label du champ', 'col' => 'col-md-3'],
                        ['name' => 'type', 'label' => 'Type', 'type' => 'select', 'options' => ['text' => 'Texte', 'email' => 'Email', 'tel' => 'Téléphone', 'textarea' => 'Zone de texte', 'select' => 'Liste'], 'col' => 'col-md-2'],
                        ['name' => 'placeholder', 'label' => 'Placeholder', 'type' => 'text', 'placeholder' => 'Texte indicatif', 'col' => 'col-md-2'],
                        ['name' => 'required', 'label' => 'Requis', 'type' => 'checkbox', 'col' => 'col-md-1'],
                    ];
                    echo render_array_items('contact_form_field', $contactFormFields, $contactFormFieldFields, 'Champ');
                    ?>
                  </div>
                  <div class="col-md-4"><label class="form-label" for="contact_form_submit">Label soumission</label><input type="text" id="contact_form_submit" name="contact_form_submit" class="form-control" value="<?= h(field('contact_form_submit', value_from($content, ['contact','form','submit_label']), $usePostValues)) ?>" /></div>
                  <div class="col-12"><label class="form-label" for="contact_form_disclaimer">Disclaimer</label><textarea id="contact_form_disclaimer" name="contact_form_disclaimer" class="form-control" rows="2"><?= h(field('contact_form_disclaimer', value_from($content, ['contact','form','disclaimer']), $usePostValues)) ?></textarea></div>
                </div>
              </div>

              <div class="section-card card mb-3">
                <div class="card-header">Footer</div>
                <div class="card-body row g-3">
                  <div class="col-md-6"><label class="form-label" for="footer_copyright">Copyright</label><input type="text" id="footer_copyright" name="footer_copyright" class="form-control" value="<?= h(field('footer_copyright', value_from($content, ['footer','copyright']), $usePostValues)) ?>" /></div>
                  <div class="col-md-6"><label class="form-label" for="footer_slogan">Slogan</label><input type="text" id="footer_slogan" name="footer_slogan" class="form-control" value="<?= h(field('footer_slogan', value_from($content, ['footer','slogan']), $usePostValues)) ?>" /></div>
                </div>
              </div>

              <div class="section-card card mb-3">
                <div class="card-header">Forms</div>
                <div class="card-body row g-3">
                  <div class="col-md-6"><label class="form-label" for="forms_mail_subject">Sujet mail</label><input type="text" id="forms_mail_subject" name="forms_mail_subject" class="form-control" value="<?= h(field('forms_mail_subject', value_from($content, ['forms','mail_subject']), $usePostValues)) ?>" /></div>
                </div>
              </div>

              <div class="sticky-top-form bg-white p-3 rounded shadow-sm mb-3">
                <div class="d-flex justify-content-between align-items-center">
                  <div>
                    <small class="text-muted"><i class="bi bi-info-circle"></i> Conseil : gardez une copie de sauvegarde avant modification.</small>
                  </div>
                  <div>
                    <button type="button" class="btn btn-outline-secondary btn-sm me-2" onclick="expandAll()"><i class="bi bi-arrows-expand"></i> Tout déplier</button>
                    <button type="button" class="btn btn-outline-secondary btn-sm me-2" onclick="collapseAll()"><i class="bi bi-arrows-collapse"></i> Tout replier</button>
                    <button type="submit" class="btn btn-brand"><i class="bi bi-check-circle me-2"></i>Enregistrer</button>
                  </div>
                </div>
              </div>
            </form>
          </div>
        </div>
      </div>
    </div>
  </div>

  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
  <script>
    function addArrayItem(id, fields, itemLabel) {
      const container = document.getElementById(id + '_container');
      if (!container) return;
      
      const existingItems = container.querySelectorAll('.array-item-card');
      const newIndex = existingItems.length;
      
      const itemCard = document.createElement('div');
      itemCard.className = 'array-item-card card mb-2';
      itemCard.setAttribute('data-index', newIndex);
      
      let html = '<div class="card-body">';
      html += '<div class="d-flex justify-content-between align-items-center mb-2">';
      html += '<small class="text-muted fw-bold">' + (itemLabel || 'Élément #' + (newIndex + 1)) + '</small>';
      html += '<button type="button" class="btn btn-sm btn-outline-danger" onclick="removeArrayItem(this)"><i class="bi bi-trash"></i></button>';
      html += '</div>';
      html += '<div class="row g-2">';
      
      fields.forEach(field => {
        const fieldName = id + '_' + newIndex + '_' + field.name;
        html += '<div class="' + (field.col || 'col-12') + '">';
        html += '<label class="form-label small">' + field.label + '</label>';
        
        if (field.type === 'textarea') {
          html += '<textarea name="' + fieldName + '" class="form-control form-control-sm" rows="' + (field.rows || 2) + '" placeholder="' + (field.placeholder || '') + '"></textarea>';
        } else if (field.type === 'select') {
          html += '<select name="' + fieldName + '" class="form-select form-select-sm">';
          if (field.options) {
            Object.keys(field.options).forEach(key => {
              html += '<option value="' + key + '">' + field.options[key] + '</option>';
            });
          }
          html += '</select>';
        } else if (field.name === 'required' || field.type === 'checkbox') {
          html += '<div class="form-check">';
          html += '<input type="checkbox" name="' + fieldName + '" class="form-check-input" value="1" />';
          html += '<label class="form-check-label small">Oui</label>';
          html += '</div>';
        } else {
          html += '<input type="' + (field.type || 'text') + '" name="' + fieldName + '" class="form-control form-control-sm" value="" placeholder="' + (field.placeholder || '') + '" />';
        }
        html += '</div>';
      });
      
      html += '</div></div></div>';
      itemCard.innerHTML = html;
      container.appendChild(itemCard);
    }
    
    function removeArrayItem(btn) {
      const itemCard = btn.closest('.array-item-card');
      if (itemCard && confirm('Supprimer cet élément ?')) {
        itemCard.remove();
        // Réindexer les éléments restants
        const container = itemCard.parentElement;
        const items = container.querySelectorAll('.array-item-card');
        items.forEach((item, index) => {
          item.setAttribute('data-index', index);
          const id = container.id.replace('_container', '');
          item.querySelectorAll('input, select, textarea').forEach(input => {
            const name = input.name;
            const parts = name.split('_');
            parts[parts.length - 2] = index;
            input.name = parts.join('_');
          });
        });
      }
    }
    
    // Fonctions pour les blocs d'activités
    let activityBlockCounter = document.querySelectorAll('.activity-block-card').length || 0;
    
    function addActivityBlock() {
      // Trouver le conteneur des blocs d'activités
      const labels = Array.from(document.querySelectorAll('label'));
      const activitiesLabel = labels.find(l => l.textContent.includes('Blocs d\'activités'));
      if (!activitiesLabel) return;
      const container = activitiesLabel.closest('.col-12');
      if (!container) return;
      
      const blockCard = document.createElement('div');
      blockCard.className = 'activity-block-card card mb-3';
      blockCard.setAttribute('data-index', activityBlockCounter);
      
      let html = '<div class="card-body">';
      html += '<div class="d-flex justify-content-between align-items-center mb-2">';
      html += '<small class="text-muted fw-bold">Bloc #' + (activityBlockCounter + 1) + '</small>';
      html += '<button type="button" class="btn btn-sm btn-outline-danger" onclick="removeActivityBlock(this)"><i class="bi bi-trash"></i> Supprimer le bloc</button>';
      html += '</div>';
      html += '<div class="row g-2 mb-3">';
      html += '<div class="col-md-8"><label class="form-label small">Titre du bloc</label>';
      html += '<input type="text" name="activity_block_' + activityBlockCounter + '_title" class="form-control form-control-sm" placeholder="Ex: A. Pour le bon goût de l\'esprit scout" /></div>';
      html += '<div class="col-md-4"><label class="form-label small">Largeur colonne</label>';
      html += '<select name="activity_block_' + activityBlockCounter + '_col" class="form-select form-select-sm">';
      html += '<option value="col-md-12" selected>Pleine largeur</option>';
      html += '<option value="col-md-6">Demi largeur</option>';
      html += '<option value="col-md-4">Tiers largeur</option>';
      html += '</select></div>';
      html += '</div>';
      html += '<div class="activity-block-items" data-block-index="' + activityBlockCounter + '"></div>';
      html += '<button type="button" class="btn btn-sm btn-outline-primary" onclick="addActivityItem(' + activityBlockCounter + ')"><i class="bi bi-plus-circle"></i> Ajouter un item</button>';
      html += '</div></div>';
      
      blockCard.innerHTML = html;
      const buttons = Array.from(container.querySelectorAll('button'));
      const addBtn = buttons.find(b => b.textContent.includes('Ajouter un bloc'));
      if (addBtn) {
        addBtn.parentElement.insertBefore(blockCard, addBtn);
      } else {
        container.appendChild(blockCard);
      }
      activityBlockCounter++;
    }
    
    function removeActivityBlock(btn) {
      const blockCard = btn.closest('.activity-block-card');
      if (blockCard && confirm('Supprimer ce bloc d\'activités ?')) {
        blockCard.remove();
      }
    }
    
    function addActivityItem(blockIndex) {
      const itemsContainer = document.querySelector('.activity-block-items[data-block-index="' + blockIndex + '"]');
      if (!itemsContainer) return;
      
      const itemIndex = itemsContainer.querySelectorAll('textarea').length;
      const itemDiv = document.createElement('div');
      itemDiv.className = 'd-flex mb-2 align-items-start';
      itemDiv.innerHTML = '<textarea name="activity_block_' + blockIndex + '_item_' + itemIndex + '" class="form-control form-control-sm me-2" rows="2" placeholder="Item d\'activité (HTML autorisé)"></textarea>';
      itemDiv.innerHTML += '<button type="button" class="btn btn-sm btn-outline-danger" onclick="removeActivityItem(this)"><i class="bi bi-x"></i></button>';
      itemsContainer.appendChild(itemDiv);
    }
    
    function removeActivityItem(btn) {
      const itemDiv = btn.parentElement;
      if (itemDiv && confirm('Supprimer cet item ?')) {
        itemDiv.remove();
      }
    }
    
    function expandAll() {
      document.querySelectorAll('.collapse').forEach(el => {
        const bsCollapse = new bootstrap.Collapse(el, {toggle: false, show: true});
      });
    }

    function collapseAll() {
      document.querySelectorAll('.collapse').forEach(el => {
        const bsCollapse = new bootstrap.Collapse(el, {toggle: false, hide: true});
      });
    }

    // Animation des chevrons
    document.addEventListener('DOMContentLoaded', () => {
      document.querySelectorAll('.section-header').forEach(header => {
        header.addEventListener('click', function() {
          const icon = this.querySelector('.bi-chevron-down, .bi-chevron-up');
          if (icon) {
            const targetId = this.getAttribute('data-bs-target');
            const target = document.querySelector(targetId);
            if (target) {
              target.addEventListener('shown.bs.collapse', () => {
                icon.classList.remove('bi-chevron-down');
                icon.classList.add('bi-chevron-up');
              });
              target.addEventListener('hidden.bs.collapse', () => {
                icon.classList.remove('bi-chevron-up');
                icon.classList.add('bi-chevron-down');
              });
            }
          }
        });
      });
    });
  </script>
</body>
</html>

